<?php
require_once __DIR__ . '/db.php';

function json_response($data, int $status = 200): void {
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

function read_json(): array {
    $raw = file_get_contents('php://input');
    if (!$raw) return [];
    $decoded = json_decode($raw, true);
    return is_array($decoded) ? $decoded : [];
}

function bearer_token(): ?string {
    $hdr = $_SERVER['HTTP_AUTHORIZATION'] ?? '';
    if (!$hdr) return null;
    if (preg_match('/Bearer\s+(.*)$/i', $hdr, $m)) return trim($m[1]);
    return null;
}

function require_vendor_auth(): array {
    $token = bearer_token();
    if (!$token) json_response(['success' => false, 'message' => 'Unauthorized'], 401);

    $pdo = db();
    $stmt = $pdo->prepare("SELECT vt.vendor_id, v.* FROM vendor_tokens vt JOIN vendors v ON v.id = vt.vendor_id WHERE vt.token = ? AND vt.expires_at > NOW() LIMIT 1");
    $stmt->execute([$token]);
    $row = $stmt->fetch();
    if (!$row) json_response(['success' => false, 'message' => 'Unauthorized'], 401);
    return $row;
}

function random_token(int $len = 48): string {
    return bin2hex(random_bytes(intdiv($len, 2)));
}

function upload_file(string $field, string $subdir): ?string {
    if (!isset($_FILES[$field]) || $_FILES[$field]['error'] !== UPLOAD_ERR_OK) return null;

    $cfg = app_config();
    $uploadRoot = realpath(__DIR__ . '/../public') . '/' . trim($cfg['UPLOAD_DIR'], '/');
    if (!$uploadRoot) $uploadRoot = __DIR__ . '/../uploads';

    $targetDir = rtrim($uploadRoot, '/') . '/' . trim($subdir, '/');
    if (!is_dir($targetDir)) mkdir($targetDir, 0777, true);

    $name = $_FILES[$field]['name'] ?? 'file';
    $ext = pathinfo($name, PATHINFO_EXTENSION);
    $safeExt = preg_replace('/[^a-zA-Z0-9]/', '', $ext);
    $filename = date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . ($safeExt ? '.' . $safeExt : '');

    $dest = $targetDir . '/' . $filename;
    if (!move_uploaded_file($_FILES[$field]['tmp_name'], $dest)) return null;

    // public path
    $publicPath = trim($cfg['UPLOAD_DIR'], '/') . '/' . trim($subdir, '/') . '/' . $filename;
    return $publicPath;
}

function like_today_or_tomorrow(string $slot): string {
    $slot = trim($slot);
    if ($slot === '') return '';
    return $slot;
}
